<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\PromotionService;
use App\Services\ServiceService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class PromotionController extends Controller
{
    protected PromotionService $promotionService;
    protected ServiceService $serviceService;

    /**
     * Available statuses for promotions.
     */
    protected array $statuses = [
        'inactive' => 'Ngừng áp dụng',
        'active' => 'Đang chạy',
        'scheduled' => 'Chờ áp dụng',
        'expired' => 'Đã kết thúc',
    ];

    public function __construct(PromotionService $promotionService, ServiceService $serviceService)
    {
        $this->promotionService = $promotionService;
        $this->serviceService = $serviceService;
    }

    /**
     * Display a listing of promotions.
     */
    public function index()
    {
        // Tự động cập nhật trạng thái khuyến mãi trước khi hiển thị
        $this->promotionService->autoUpdateAllPromotionStatuses();
        
        $promotions = $this->promotionService->getAll();
        $statuses = $this->statuses;
        $isTrash = false;

        return view('admin.promotions.index', compact('promotions', 'statuses', 'isTrash'));
    }

    /**
     * Display a listing of trashed promotions.
     */
    public function trash()
    {
        $promotions = $this->promotionService->getTrashed();
        $statuses = $this->statuses;
        $isTrash = true;

        return view('admin.promotions.index', compact('promotions', 'statuses', 'isTrash'));
    }

    /**
     * Show the form for creating a new promotion.
     */
    public function create()
    {
        $statuses = $this->statuses;
        // Chỉ lấy danh sách dịch vụ đơn để tránh trộn với biến thể
        $services = $this->serviceService->getBaseServices();
        $combos = \App\Models\Combo::whereNull('deleted_at')->with('category')->get();
        $serviceVariants = \App\Models\ServiceVariant::whereNull('deleted_at')
            ->with('service.category')
            ->get();

        return view('admin.promotions.create', compact('statuses', 'services', 'combos', 'serviceVariants'));
    }

    /**
     * Store a newly created promotion.
     */
    public function store(Request $request)
    {
        $data = $this->validateData($request);
        $serviceIds = $request->input('services', []);
        $comboIds = $request->input('combos', []);
        $variantIds = $request->input('service_variants', []);
        
        $this->promotionService->create($data, $serviceIds, $comboIds, $variantIds);

        return redirect()
            ->route('admin.promotions.index')
            ->with('success', 'Khuyến mãi đã được tạo thành công!');
    }

    /**
     * Display the specified promotion.
     */
    public function show(string $id)
    {
        $promotion = $this->promotionService->getOne($id);
        $statuses = $this->statuses;

        return view('admin.promotions.show', compact('promotion', 'statuses'));
    }

    /**
     * Show the form for editing the specified promotion.
     */
    public function edit(string $id)
    {
        $promotion = $this->promotionService->getOne($id);
        $statuses = $this->statuses;
        // Chỉ lấy danh sách dịch vụ đơn để tránh trộn với biến thể
        $services = $this->serviceService->getBaseServices();
        $combos = \App\Models\Combo::whereNull('deleted_at')->with('category')->get();
        $serviceVariants = \App\Models\ServiceVariant::whereNull('deleted_at')
            ->with('service.category')
            ->get();
        $selectedServiceIds = $promotion->services->pluck('id')->toArray();
        $selectedComboIds = $promotion->combos->pluck('id')->toArray();
        $selectedVariantIds = $promotion->serviceVariants->pluck('id')->toArray();

        return view('admin.promotions.edit', compact(
            'promotion', 
            'statuses', 
            'services', 
            'combos',
            'serviceVariants',
            'selectedServiceIds',
            'selectedComboIds',
            'selectedVariantIds'
        ));
    }

    /**
     * Update the specified promotion in storage.
     */
    public function update(Request $request, string $id)
    {
        $data = $this->validateData($request, $id);
        $serviceIds = $request->input('services', []);
        $comboIds = $request->input('combos', []);
        $variantIds = $request->input('service_variants', []);
        
        $this->promotionService->update($id, $data, $serviceIds, $comboIds, $variantIds);

        return redirect()
            ->route('admin.promotions.index')
            ->with('success', 'Khuyến mãi đã được cập nhật!');
    }

    /**
     * Remove the specified promotion from storage (soft delete).
     */
    public function destroy(string $id)
    {
        $this->promotionService->delete($id);

        return redirect()
            ->route('admin.promotions.index')
            ->with('success', 'Khuyến mãi đã được xóa!');
    }

    /**
     * Restore the specified trashed promotion.
     */
    public function restore(string $id)
    {
        $this->promotionService->restore($id);

        return redirect()
            ->route('admin.promotions.trash')
            ->with('success', 'Khuyến mãi đã được khôi phục!');
    }

    /**
     * Permanently delete the specified promotion.
     */
    public function forceDelete(string $id)
    {
        $this->promotionService->forceDelete($id);

        return redirect()
            ->route('admin.promotions.trash')
            ->with('success', 'Khuyến mãi đã được xóa vĩnh viễn!');
    }

    /**
     * Validate request data.
     */
    protected function validateData(Request $request, ?string $id = null): array
    {
        $data = $request->validate([
            'code' => [
                'required',
                'string',
                'max:50',
                Rule::unique('promotions', 'code')->ignore($id),
            ],
            'name' => ['required', 'string', 'max:191'],
            'description' => ['nullable', 'string'],
            'discount_type' => ['required', Rule::in(['percent', 'amount'])],
            'discount_percent' => ['nullable', 'integer', 'min:0', 'max:100'],
            'discount_amount' => ['nullable', 'numeric', 'min:0'],
            'apply_scope' => ['required', Rule::in(['service', 'order', 'customer_tier'])],
            'min_order_amount' => ['nullable', 'numeric', 'min:0'],
            'max_discount_amount' => ['nullable', 'numeric', 'min:0'],
            'per_user_limit' => ['nullable', 'integer', 'min:1'],
            'usage_limit' => ['nullable', 'integer', 'min:1'],
            'min_customer_tier' => ['nullable', Rule::in(['Khách thường', 'Silver', 'Gold', 'VIP'])],
            'start_date' => ['required', 'date'],
            'end_date' => ['nullable', 'date', 'after_or_equal:start_date'],
            'status' => ['required', Rule::in(array_keys($this->statuses))],
        ]);

        // Ràng buộc logic theo loại giảm giá
        if ($data['discount_type'] === 'percent') {
            $data['discount_amount'] = null;
            if ($data['discount_percent'] === null || $data['discount_percent'] <= 0) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'discount_percent' => 'Vui lòng nhập % giảm giá lớn hơn 0 khi chọn loại giảm giá theo phần trăm.',
                ]);
            }
            if ($data['discount_percent'] > 100) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'discount_percent' => '% giảm giá không được vượt quá 100%.',
                ]);
            }
        } else {
            // Set discount_percent = 0 thay vì null để tránh lỗi database
            $data['discount_percent'] = 0;
            if ($data['discount_amount'] === null || $data['discount_amount'] <= 0) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'discount_amount' => 'Vui lòng nhập số tiền giảm giá lớn hơn 0 khi chọn loại giảm giá theo số tiền.',
                ]);
            }
        }

        // Nếu áp dụng theo hóa đơn thì yêu cầu tổng tiền tối thiểu
        if ($data['apply_scope'] === 'order' && ($data['min_order_amount'] === null || $data['min_order_amount'] <= 0)) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'min_order_amount' => 'Vui lòng nhập số tiền hóa đơn tối thiểu lớn hơn 0 khi áp dụng theo hóa đơn.',
            ]);
        }

        // Nếu áp dụng theo hạng khách hàng thì yêu cầu chọn hạng tối thiểu
        if ($data['apply_scope'] === 'customer_tier' && empty($data['min_customer_tier'])) {
            throw \Illuminate\Validation\ValidationException::withMessages([
                'min_customer_tier' => 'Vui lòng chọn hạng khách hàng tối thiểu khi áp dụng theo hạng khách hàng.',
            ]);
        }

        // Nếu áp dụng theo dịch vụ thì yêu cầu chọn ít nhất một dịch vụ/combo/variant
        if ($data['apply_scope'] === 'service') {
            $serviceIds = $request->input('services', []);
            $comboIds = $request->input('combos', []);
            $variantIds = $request->input('service_variants', []);
            
            if (empty($serviceIds) && empty($comboIds) && empty($variantIds)) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'services' => 'Vui lòng chọn ít nhất một dịch vụ, combo hoặc dịch vụ biến thể khi áp dụng theo dịch vụ.',
                ]);
            }
        }

        return $data;
    }
}

